//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI

struct TodoList: View {
    private enum SortOption: String, CaseIterable {
        case title = "Title"
        case priority = "Priority"
        case dueDate = "Due Date"
    }
    
    @State private var sortBy: SortOption = .title
    @State private var showingChooser: Bool = false

    @EnvironmentObject private var data: DataCenter

    var list: TodoItemList? = nil
    var items: [TodoItem] {
        guard let list = list else { return data.todoItems }
        return data.items(in: list)
    }
    var title: String { list?.name ?? "All Items" }

    func color(for item: TodoItem) -> Color {
        if let list = self.list {
            return list.color.uiColor
        }
        return data.list(for: item).color.uiColor
    }
    
    var sortButton: some View {
        Button(action: {
            self.showingChooser.toggle()
        }) {
            Image(systemName: "arrow.up.arrow.down.square")
                .imageScale(.large)
                .font(.system(size: 24, weight: .bold))
        }
    }

    var body: some View {
        List(sortedItems) { item in
            NavigationLink(destination: TodoItemDetail(item: item)
                .environmentObject(self.data)
            ) {
                TodoItemRow(item: item)
                    .accentColor(self.color(for: item))
            }
        }
        .navigationBarTitle(LocalizedStringKey(title))
        .navigationBarItems(trailing: sortButton)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(LocalizedStringKey(opt.rawValue))) {
                        self.sortBy = opt
                    }
            })
        }
    }
    
    private var sortedItems: [TodoItem] {
        items.sorted {
            switch sortBy {
            case .title:
                return $0.title.lowercased() < $1.title.lowercased()
            case .priority:
                return $0.priority > $1.priority
            case .dueDate:
                return ($0.date ?? .distantFuture) <
                    ($1.date ?? .distantFuture)
            }
        }
    }
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        TodoList()
            .environmentObject(DataCenter())
    }
}
